/*********************************************************************
 *                         DISK CRC CALCULATION                      *
 *********************************************************************
 *
 * Project:       DiskCRC
 * Type:          Source file
 * Description:   Source file for disk CRC calculation program
 * Author:		  John Wyatt	
 *
 *********************************************************************
 * Copyright(c) 2004 by John Wyatt.                                  *
 * All rights reserved                                               *
 *********************************************************************

  Purpose:		To compute a CRC on an entire directory tree of files.
				This is used to verify distribution disks to ensure
				they were not corrupted.

  Key Objects:

	m_FileListBox	A listbox object that displays the files checked
					and the results. After the calculations, it displays
					the number of total files, and the CRC. It also
					displays a message indicating if the CRC matched
					the expected CRC.

	mSortedDirBox	A hidden listbox that is different from the
					m_FileListBox in that it has the "sorted" attribute
					set. This is the quick way to get a sorted directory
					list. I couldn't set the sort attribute of
					m_FileListBox, or the status lines at the end would
					be sorted and appear somewhere in the middle.
					I didn't want to write a bubble sort routine,
					so the files are entered into this hidden control
					and sorted for me. Once filled, I copy them to
					a CStringList. 

	m_strDirList	CStringList object that holds the full pathnames
					of the files to be processed.

	m_uActualCRC	The 32-bit CRC. This CRC is accumulated as each
					file is processed.

	m_crcEdit		Edit control used to enter the expected CRC. This 
					control gets set to read-only by the checkbox
					m_CRClocked. The purpose of locking it is to 
					prevent accidental alteration if one is testing
					many distribution disks.

	Program Flow:
		Get Root Directory from the Command Line
		Get the expected CRC
		Make a sorted list of all files

		FOR each file DO
			Calculate cumulative CRC
		END FOR

		Compare calculated CRC to the expected CRC
		Display Results

*********************************************************************/

#include "stdafx.h"
#include "filterededit.h"
#include "DiskCRC.h"
#include "DiskCRCDlg.h"
#include "AboutDialog.h"
#include "folderdialog.h"
#include "textprinter.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// Registry keys for storing the uesr settings
#define REGSECTION "USERSETTINGS"
#define REGDIRENTRY "SAVEDPATH"

/////////////////////////////////////////////////////////////////////////////
// CDiskCRCDlg dialog

CDiskCRCDlg::CDiskCRCDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CDiskCRCDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CDiskCRCDlg)
	m_bCRCUnknown = FALSE;
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

void CDiskCRCDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CDiskCRCDlg)
	DDX_Control(pDX, IDC_SORTLISTBOX, m_SortedDirBox);
	DDX_Control(pDX, IDC_CRC_EDIT, m_CRCedit);
	DDX_Control(pDX, IDC_FILELISTBOX, m_FileListBox);
	DDX_Check(pDX, IDC_CRC_UNKNOWN, m_bCRCUnknown);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CDiskCRCDlg, CDialog)
	//{{AFX_MSG_MAP(CDiskCRCDlg)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_SETDIR, OnSetdir)
	ON_BN_CLICKED(IDC_PRINTBTN, OnPrintbtn)
	ON_BN_CLICKED(IDC_CRC_UNKNOWN, OnCrcUnknown)
	ON_BN_CLICKED(IDC_RADIO_BYDISK, OnRadioBydisk)
	ON_BN_CLICKED(IDOK, OnOK)
	ON_BN_CLICKED(IDC_RADIO_BYFILE, OnRadioByfile)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDiskCRCDlg message handlers

BOOL CDiskCRCDlg::OnInitDialog()
{


	CDialog::OnInitDialog();

	// Add "About..." menu item to system menu.

	// IDM_ABOUTBOX must be in the system command range.
	ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
	ASSERT(IDM_ABOUTBOX < 0xF000);

	CMenu* pSysMenu = GetSystemMenu(FALSE);
	if (pSysMenu != NULL)
	{
		CString strAboutMenu;
		strAboutMenu.LoadString(IDS_ABOUTBOX);
		if (!strAboutMenu.IsEmpty())
		{
			pSysMenu->AppendMenu(MF_SEPARATOR);
			pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
		}
	}

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon
	
	// Get the path being checked
	// GET THE PERSISTENT SETTING FOR PATH AND STORE IN m_strRootDir
	m_strRootDir = AfxGetApp()->GetProfileString(REGSECTION, REGDIRENTRY);

	// Set window title to path being checked
	CString strTitle;
	strTitle.LoadString(IDS_WINDOWTITLE);
	strTitle += m_strRootDir;
	SetWindowText(strTitle);

	// Create the CRC object
	m_bCRCUnknown = FALSE;

	// Limit edit control to 8 chars in length
	m_CRCedit.SetLimitText(8);

	// Set to Whole Disk
	CheckRadioButton(IDC_RADIO_BYDISK, IDC_RADIO_BYFILE, IDC_RADIO_BYDISK);
	
	return TRUE;  // return TRUE  unless you set the focus to a control
}

/////////////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
	if ((nID & 0xFFF0) == IDM_ABOUTBOX)
	{
		// Show the About box
		CAboutDialog dlgAbout;
		dlgAbout.DoModal();
	}
	else
	{
		CDialog::OnSysCommand(nID, lParam);
	}
}


/////////////////////////////////////////////////////////////////////////////
// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CDiskCRCDlg::OnPaint() 
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, (WPARAM) dc.GetSafeHdc(), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialog::OnPaint();
	}
}


/////////////////////////////////////////////////////////////////////////////
// The Compute CRC button.
void CDiskCRCDlg::OnOK() 
{
	// IF a directory has never been set THEN
	while (m_strRootDir.GetLength() == 0)
	{
		// Inform use of the error and prompt for the directory.
		CString strError;
		strError.LoadString(IDS_NODIRSEL);
		AfxMessageBox(strError);
		OnSetdir();
	}

	// Set the hourglass cursor.
	// The object's constructor and destructor do all the work.
	CWaitCursor	cWait;

	// Get the expected CRC from the edit control
	m_CRCedit.GetWindowText(m_expectedCRC);


	// Force to upper case
	m_expectedCRC.MakeUpper();
	m_CRCedit.SetWindowText(m_expectedCRC);

	//-----------------------------------------------------------
	// IF an expected CRC is required, then verify that it exists.
	// IF CRC was known AND no expected CRC was entered AND checking Whole Disk THEN
	if ( (m_expectedCRC.GetLength() < 8) && !m_bCRCUnknown && IsDlgButtonChecked(IDC_RADIO_BYDISK) )
	{
		// Inform the user of the error.
		CString strMsg;
		strMsg.LoadString(IDS_INPUTERROR);
		AfxMessageBox(strMsg);
		return;		// <<<----------- ABORT -------------------
	}

	GetDlgItem(IDOK) -> SetFocus();

	//-----------------------------------------------------------
	//	Make list of all files in the directory tree
	if (!GetDirTree()) return;

	//-----------------------------------------------------------
	//	Reset CRC calculation
	m_uActualCRC = m_MyCRC.ResetCRC();

	// For each file in the list, accumulate the CRC
	// Start at the head of the file list

	for (m_Pos = m_strDirList.GetHeadPosition(); m_Pos != NULL;)
	{
		// Get filename and step to the next file
		m_strFileName = m_strDirList.GetNext(m_Pos);

		CString strDisplay = m_strFileName;

		// Calculate the CRC
		m_uActualCRC = m_MyCRC.CalcFileCRC(m_strFileName);

		// IF we are doing CRC By File, THEN
		if (IsDlgButtonChecked(IDC_RADIO_BYFILE) > 0)
		{
			// Append the CRC to the file name
			strDisplay += " - ";
			strDisplay += m_MyCRC.GetCrcString();

			// Reset the CRC
			m_MyCRC.ResetCRC();
		}
		// END IF

		// Transfer filename to the display listbox,
		// but minus the root directory path. Just insert the filenames.
		int iLen = strDisplay.GetLength();
		int iRootLen = m_strRootDir.GetLength();
		strDisplay = strDisplay.Right(iLen - iRootLen - 1);
		m_FileListBox.AddString(strDisplay);
	}

	// Display the summary statistics and whether the CRC matches or not.
	ShowSummary();

	// Reset list box to show the most recently added lines
	m_FileListBox.SetTopIndex(m_FileListBox.GetCount()-1);

	// Now that there's something to print, Enable the Print button
	GetDlgItem(IDC_PRINTBTN)->EnableWindow(TRUE);
}


/////////////////////////////////////////////////////////////////////////////
// Compares the actual CRC to the computed CRC and displays the results.
void CDiskCRCDlg::ShowSummary()
{

	// Add trailers to list box and scroll to bottom
	m_FileListBox.AddString("--------------------------------------------------");
	m_FileListBox.SetTopIndex(m_FileListBox.GetCount()-1);


	// Display total number of files
	CString strLine;
	CString ActualCRC;
	strLine.Format(IDS_TOTALFILES, m_strDirList.GetCount());
	m_FileListBox.AddString(strLine);

	// IF we are doing the whole disk THEN
	if (IsDlgButtonChecked(IDC_RADIO_BYDISK) > 0)
	{
		// Display the cumulative CRC as a hex string
		ActualCRC = m_MyCRC.GetCrcString();
		strLine.Format(IDS_ACTUALCRC, ActualCRC);
		m_FileListBox.AddString(strLine);

		// IF the expected CRC was entered THEN
		if (!m_bCRCUnknown)
		{
			// Display the expected CRC
			strLine.Format(IDS_EXPECTEDCRC, m_expectedCRC);
			m_FileListBox.AddString(strLine);

			//	Compare CRC with expected and indicate Pass/Fail
			if (ActualCRC.CompareNoCase(m_expectedCRC) == 0)
			{
				// Strings equal, CRC matches.
				strLine.LoadString(IDS_CRCMATCH);
			}
			else
			{
				// Strings not equal, CRC fails
				strLine.LoadString(IDS_CRCNOMATCH);
			}
			m_FileListBox.AddString(strLine);
		} // END IF
	} // END IF

	
}


/////////////////////////////////////////////////////////////////////////////
// Used by GetDirTree().
void CDiskCRCDlg::GetFilesInDir(CString& szDir)
{
//	INPUT:		szDir holds the directory name
//	RETURNS:	None
//				Fills m_SortDirBox with the filenames (full path)
//				These filenames will be sorted.
//				Recursion is used to read the entire directory tree.

	CFileFind	ffFiles;
	CString		strNewDir;
	CString		strCurDir;

	// Default to *.* 
	strCurDir = szDir + "*.*";
	
	// Start the file search
	BOOL bMoreFiles = ffFiles.FindFile(strCurDir);

	// Retrieve info on each file
	while (bMoreFiles)
	{
		bMoreFiles = ffFiles.FindNextFile();
		// Do not add parent or own directory ( . or .. )
		if (ffFiles.IsDots()) continue;

		// If file is a directory, recurse into directory,
		// else add it to the listbox
		if (ffFiles.IsDirectory())
		{
			// Make the new path and recurse into it
			strNewDir = ffFiles.GetFilePath() + "\\";
			GetFilesInDir(strNewDir);
		}
		else
		{
			// Add filename to the sorted list box.
			// Later, the sorted files will be transferred
			// to a CStringList.
			m_SortedDirBox.AddString(
				(LPCTSTR) ffFiles.GetFilePath());

		} // end if IsDirectory()

	} // end while

	ffFiles.Close();
}


/////////////////////////////////////////////////////////////////////////////
// Builds a sorted directory tree in the listbox.
// File CRCs will be calculated in the sorted order.
BOOL CDiskCRCDlg::GetDirTree()
{
//	INPUT:		m_strRootDir holds the root directory name
//	RETURNS:	TRUE if files were found, FALSE if no files.
//				Fills m_strDirList with the filenames (full path)
//				Clears the listbox m_FileListBox

// This function calls GetFilesInDir() to get the directory tree,
// plus performs some error checking.
// Displays the directory tree in the listbox.

	// Start by clearing the listbox on screen, 
	// and clearing the list of files.
	m_FileListBox.ResetContent();
	m_SortedDirBox.ResetContent();
	m_strDirList.RemoveAll();
	
	// Now get the list of all files in the directory tree.
	GetFilesInDir(m_strRootDir);

	// Check to see if there are any files in the list box
	int iNumFiles = m_SortedDirBox.GetCount();
	if (iNumFiles == 0)
	{
		// Display error and exit if not files in listbox
		CString strError;
		strError.LoadString(IDS_NOFILES);
		m_FileListBox.AddString(strError);
		return FALSE;
	}

	// There are files, so transfer the sorted listbox
	// over to the CString list.
	// While the files could have been sorted using a bubble sort,
	// or sorted files read directly out of the listbox,
	// I prefer processing the file reads from a CStringList object
	// rather than a listbox, and don't want to re-invent the sort.
	// I use a sorted listbox to do the sorting, then copy
	// it over to the CStringList.
	CString strTemp;
	for (int i=0; i < iNumFiles; i++)
	{
		m_SortedDirBox.GetText(i, strTemp);
		m_strDirList.AddTail(strTemp);
	}

	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// OTHER MESSAGE HANDLERS
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
// The system calls this to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CDiskCRCDlg::OnQueryDragIcon()
{
	return (HCURSOR) m_hIcon;
}

/////////////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnCancel() 
{
	
	CDialog::OnCancel();
}


/////////////////////////////////////////////////////////////////////////////
// Click handler for the Select Directory button.
// Set the drive/directory to check
void CDiskCRCDlg::OnSetdir() 
{
	CString strCaption;
	strCaption.LoadString(IDS_SELECTDIRCAPTION);

	// Get new path
	LPCSTR pszNewDir = BrowseForDirectory(m_strRootDir, strCaption);
	if (pszNewDir != NULL)
	{
		// Set the new path
		m_strRootDir = pszNewDir;

		// Also, change the window title.
		CString strTitle;
		strTitle.LoadString(IDS_WINDOWTITLE);
		strTitle += m_strRootDir;
		SetWindowText(strTitle);

		// Save the persistent setting.
		// Save the directory setting
		AfxGetApp()->WriteProfileString(REGSECTION, REGDIRENTRY, m_strRootDir);
	}

	GetDlgItem(IDOK) -> SetFocus();

}


///////////////////////////////////////////////////////////////////////
// Allows user to browse among the existing directories and select one.
// Used to set the path to verify.
//	RETURNS:	New directory.
//				IF Cancel was selected, returns NULL.
LPCSTR CDiskCRCDlg::BrowseForDirectory(LPCSTR szCurDir, LPCSTR szCaption)
{
	static	CString	szNewDir;
	LPCSTR	lpRetVal = NULL;

	// If a path was passed in, start from there.
	if (szCurDir != NULL)
	{
		szNewDir = szCurDir;
	}
	else
	{
		szNewDir.Empty();
	}

	// User is presented with a path folder and can browse to
	// select a directory.
	CFolderDialog	dlgPath(szCaption, szNewDir, 0, (CWnd*) this);

	int iResult = dlgPath.DoModal();
	if ( iResult == IDOK)
	{
		// Enter the selected directory
		szNewDir = dlgPath.GetPathName();
		lpRetVal = szNewDir;
	}

	return lpRetVal;
	
}

///////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnPrintbtn() 
{
	static	LOGFONT	DefaultLogfont;

	// The default font.
	memset(&DefaultLogfont, 0, sizeof(DefaultLogfont));
	DefaultLogfont.lfHeight = 10;
	DefaultLogfont.lfWeight = FW_NORMAL;
	DefaultLogfont.lfCharSet = DEFAULT_CHARSET;
	DefaultLogfont.lfOutPrecision = OUT_TT_PRECIS;
	DefaultLogfont.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	DefaultLogfont.lfQuality = PROOF_QUALITY;
	DefaultLogfont.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;
	_tcscpy(DefaultLogfont.lfFaceName, _T("Arial") );

	// Set the hourglass cursor.
	// The object's constructor and destructor do all the work.
	CWaitCursor	cWait;
	CString strLine;

	// Print the CRC results. Use the default printer font.
	// The constructor will create the printer.
	CTextPrinter	oPrinter( "CRC Calculation", &DefaultLogfont, &DefaultLogfont, &DefaultLogfont, 2, 2);
	if (!oPrinter.IsSelected()) return;	// <<-- EXCEPTION -- User did not select a printer!

	// Set the header to the software's name and version.
	CAboutDialog dlgAbout;
	strLine = dlgAbout.GetProduct();
	strLine += " ";
	strLine += dlgAbout.GetVersion();
	oPrinter.SetHeader(strLine);

	// Set the column header to be the root directory under calculation.
	strLine.LoadString(IDS_PAGEHEADER);
	strLine += m_strRootDir;
	oPrinter.SetLegend(strLine, 0);

	// Determine how many pages will be needed.
	// Although the CTextPrinter object automatically page breaks as needed,
	// the footer line is nicer if it says "Page X of Y",
	// instead of just "Page X". To do that, we must know in advance
	// how many pages will be needed.
	int iNumLinesNeeded = m_FileListBox.GetCount();
	int iNumLinesPerPage = oPrinter.GetLinesPerPage();
	int iNumPages = (iNumLinesNeeded / iNumLinesPerPage ) + 1;
	oPrinter.SetNumPages(iNumPages);
	
	// Now print the contents of the listbox.
	//FOR each line in the output list box DO
	for (int iCur = 0; iCur < iNumLinesNeeded; iCur++)
	{
		// Get the line
		m_FileListBox.GetText(iCur, strLine);

		// Print the line.
		oPrinter.PrintLn(strLine);
	}
	// END FOR
}

///////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnCrcUnknown() 
{
	// Enable or disable the Expected CRC edit box based on this.
	UpdateData(TRUE);

	GetDlgItem(IDC_CRC_EDIT)->EnableWindow( (m_bCRCUnknown) ? FALSE : TRUE );
}

///////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnRadioBydisk() 
{
	// Enable the appropriate controls
	UpdateData(TRUE);
	GetDlgItem(IDC_CRC_EDIT)->EnableWindow( (m_bCRCUnknown) ? FALSE : TRUE );
	GetDlgItem(IDC_CRC_UNKNOWN)->EnableWindow( TRUE );
	GetDlgItem(IDC_STATICCRC)->EnableWindow( TRUE );
	
}

///////////////////////////////////////////////////////////////////////
void CDiskCRCDlg::OnRadioByfile() 
{
	// For a File by File CRC, we don't need an expected disk CRC
	// So disable those controls.
	UpdateData(TRUE);
	GetDlgItem(IDC_CRC_EDIT)->EnableWindow( FALSE );
	GetDlgItem(IDC_CRC_UNKNOWN)->EnableWindow( FALSE );
	GetDlgItem(IDC_STATICCRC)->EnableWindow( FALSE );
	
}
