/*********************************************************
MULTINSTANCE.CPP

  Author: John Wyatt

  Source file for object to detect a multiple instance
  of the application.

  Multiple instances are detected by created a mutex object.
  A second instance will be able to see that the mutex
  already exists, therefore a previous instance exists.

  TO USE THIS OBJECT:
	Add the files MULTINSTANCE.CPP and MULTINSTANCE.H
  
	To the file that contains the app object, include
	multinstance.h

	To the String resource, add a string that defines the
	name of the mutex.

	To the file that contains the app object, declare
	a global scope variable of type CMultInstance, with
	the name of the mutex object to create. The name can
	be either a string name of no more than MAX_PATH chars,
	or the resource ID of an entry in the string table.
	Example:
	CMultInstance MyInstance("MyInstance");

	In the applications InitInstance event, add a call to
	IsOnlyInstance()
	The call will return FALSE if this is not the only instance, 
	in which case InitInstance should return FALSE.

  EXCEPTIONS THROWN:
	In debug mode, an ASSERT fails if the mutex name
	is longer than MAX_PATH characters long or is of zero length.

	Memory exception if IsOnlyInstance() cannot create the mutex.

	Resource exception if the string IDs cannot be loaded.

  HOW THIS OBJECT WORKS
	The object is intended to be global in scope, so it
	is constucted at program start and destroyed at termination.
	A good place to check it is in WinMain 
	(or in the applications InitInstance() for MFC).

  //------------------------------------------------
	- Construction:
	Set my private handle for the mutex to NULL
	IF constructed with a string name THEN
		Store the string name
	ELSE IF constructed with a string table resource ID THEN
		Store the resource ID
	ENDIF

  //------------------------------------------------
	- Call IsOnlyInstance():
	IF the mutex name is in the string table THEN
		Get the name from the string table
		IF string cannot be read THEN
			Throw a resource exception
		ENDIF
	ENDIF

	Attempt to create the mutex

	SELECT (The result was)
		CASE A new Mutex was successfully created
			No previous instance exists
			Set my private handle to the mutex
			Return TRUE

		CASE The Mutex already exists
			There is a previous instance
			Set my private handle to the mutex
			RETURN FALSE

		CASE Mutex cannot be created
			Throw a memory exception
	END SELECT

  //------------------------------------------------
	- Call Invoke( )
	If desired, once it has been determined that a prior instance exists,
	the caller can call Invoke(...) with the string in its title bar.
	This uses the WIN32 API to enumerate through all open windows
	until it finds a match, then makes it the topmost window. 
	The logic is:
	FOR each open window
		IF this is a match for my title bar THEN
			Activate that window and bring it to the top.
			Break out of FOR loop.
		END IF
	END FOR


  //------------------------------------------------
	- Destruction:
	IF my private handle is valid THEN
		Close my mutex handle
		IF I was the only owner of the mutex THEN
			Windows will destroy the mutex automatically
		ENDIF
	ENDIF

*********************************************************/

#include "stdafx.h"
#include "MultInstance.h"
#include <string.h>


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//////////////////////////////////////////////////////////
// Constructor
// Initialize instance handle to the mutex.
// If in debug mode, verify that the mutex name is a valid length.
//////////////////////////////////////////////////////////
CMultInstance::CMultInstance(TCHAR* szMyName)
{
	m_hMyInstance = NULL;
	m_uStringID = 0;
	m_bInStringTable = FALSE;

	#ifdef _DEBUG
		ASSERT(szMyName != NULL);
		int iLength = _tcsclen(szMyName);
		ASSERT(iLength <= MAX_PATH);
		ASSERT(iLength > 0);
	#endif

	_tcscpy(m_szMutexName, szMyName);
}


//////////////////////////////////////////////////////////
// Alternate Constructor
// Initialize instance handle to the mutex.
// If in debug mode, verify that the name ID is valid.
// Save the string resource ID for later use. Cannot use
// it now, as the app will not yet have an instance handle 
// if this is a global object, so the resources can't be read yet.
//////////////////////////////////////////////////////////
CMultInstance::CMultInstance(UINT iMyNameID)
{
	m_hMyInstance = NULL;
	m_szMutexName[0] = _T('\0');
	ASSERT( iMyNameID != 0);
	m_uStringID = iMyNameID;
	m_bInStringTable = TRUE;

}

//////////////////////////////////////////////////////////
// Destructor
// If the mutex was created, destroy it.
// CloseHandle decrements the use count and deletes the
// mutex when the count reaches zero.
//////////////////////////////////////////////////////////
CMultInstance::~CMultInstance()
{
	if ( m_hMyInstance != NULL ) {
		CloseHandle(m_hMyInstance);
	}
}

//////////////////////////////////////////////////////////
// Test for multiple instance
// Returns TRUE if this is the only instance of the app.
// Throws a resource exception if the mutex name is in 
// the string table, but cannot be loaded.
// Throws a Memory Exception if the mutex cannot be
// created at all.
// There is a second check for a valid mutex name length,
// in case it was wrong in the string table.
//////////////////////////////////////////////////////////
BOOL CMultInstance::IsOnlyInstance()
{

	// If a string resource ID was used for the mutex name,
	// get the string
	if ( m_bInStringTable == TRUE ) {
		int iReadOK = LoadString (
					AfxGetResourceHandle(),
					m_uStringID,
					m_szMutexName,
					sizeof(m_szMutexName));
		if (iReadOK == 0){
			AfxThrowResourceException();
		}

		#ifdef _DEBUG
			int iLength = _tcsclen(m_szMutexName);
			ASSERT(iLength <= MAX_PATH);
			ASSERT(iLength > 0);
		#endif
	} // End If reading name from string table

	// We have a valid mutex name. Let's see if we can create one.
	m_hMyInstance = CreateMutex(NULL, TRUE, m_szMutexName);
	if ( m_hMyInstance == NULL)	{
		AfxThrowMemoryException( );
	}

	// Has a prior instance created the mutex already?
	DWORD dwError = GetLastError();
	if ( dwError == ERROR_ALREADY_EXISTS ) {
		return FALSE;
	}
	else {
		return TRUE;
	}

} // end IsOnlyInstance()

//////////////////////////////////////////////////////////
//	Invokes the application whose window titlebar
//	contains the text in the string.
void CMultInstance::Invoke(LPCSTR szName)
{
	// Find and activate the existing Worklist Editor.
	EnumWindows(&CMultInstance::EnumWindowsProc, (LPARAM) szName );
}

//////////////////////////////////////////////////////////
//	Invokes the application whose window titlebar
//	contains the text in the string resource uID.
void CMultInstance::Invoke(UINT uID)
{
	static TCHAR	szBuffer[MAX_PATH];

	int iReadOK = LoadString (
				AfxGetResourceHandle(),
				uID,
				szBuffer,
				sizeof(szBuffer) - 1);
	if (iReadOK == 0){
		AfxThrowResourceException();
	}

	Invoke(szBuffer);
}

//---------------------------------------------------------------------------
/////////////////////////////////////////////////////////////////////////////
//	Callback iterator for EnumWindows() to find a specific window.
//	Used to activate the existing Worklist Editor if a multiple instance
//	is created.
//	RETURNS:	FALSE to stop enumeration when it finds the window
//				specified by the string lParam.
BOOL CALLBACK CMultInstance::EnumWindowsProc(
  HWND hwnd,      // handle to parent window passed to this iteration
  LPARAM lParam   // application-defined value, in this case a string.
)
{
	BOOL	bRetVal = TRUE;
	TCHAR	szWindowText[256];
	LPCSTR	pszText = (LPCSTR) lParam;

	// Get the window text for this window.
	int iChars = GetWindowText(hwnd, szWindowText, sizeof(szWindowText) - 1 );
	if (iChars > 1)
	{
		// Is this the window we want?
		CString szWindow = szWindowText;
		int iPos = szWindow.Find(pszText);
		if (iPos >= 0)
		{
			// This is the intended window!
			// Make it active and stop the enumeration.
			SetForegroundWindow(hwnd);
			bRetVal = FALSE;
		}
	}

	return bRetVal;
}