//////////////////////////////////////////////////////////////////////
// Printer.cpp: implementation of the CTextPrinter class.
//
//--------------------------------------------------------------------
//	USED BY ANY OBJECT THAT WANTS TO PRINT
//--------------------------------------------------------------------
//
//	This text printer is a basic line printer with a header and footer.
//	This object is created to start a specific print job.
//	Printer errors are also handled by this object.
//
//	The constructor prompts the user to select the destination printer.
//	The destructor cleans up the printing and closes the printer.
//	To use this object, either:
//		a) Declare it locally within a function, or
//		b) Use new CTextPrinter, and delete it when done.
//
//--------------------------------------------------------------------
//	OVERRIDABLES:
//		PrintFooter()
//		PrintHeader()
//
//	Derived classes can define their own header and footer if they
//	don't want to use the default header and footer.
//
//--------------------------------------------------------------------
//	RESPONSIBILITIES
//
//	The basic workflow to perform printing is:
//		User selects a printer
//		Get the printer device context
//		Open the document
//		FOR each page
//			Start Page
//			Print page header, data, and footer
//			End Page
//		END FOR
//		End Document
//
//	Constructor
//		Prompts user to select a printer.
//		The user can cancel the print during selection.
//		If selected, calculate the page dimensions, margins,
//		number of lines per page, etc.
//		Creates the fonts for the header, body, and footer.
//
//	IsSelected()
//		Returns TRUE if the user selected a printer and no error has occurred.
//
//	Destructor
//		Finishes any pending print jobs.
//		This includes doing the EndPage() and EndDoc() as needed to
//		complete the print job.
//
//	Tab Stops
//		Provides tab stops in the form of tabbing to a specific position..
//		This allows text to be printed starting at an absolute line or column,
//		as defined by the caller.
//
//	PAGELAYOUT.H
//		Header file that lays out the margins, line spacing, etc.
//
//
//--------------------------------------------------------------------
//	PAGE LAYOUT
//
//	Pagelayout.h defines the margins in inches and the line spacing
//	as a multipler. e.g. 0.5 spacing means half of a line height as
//	additional whitespace beteen lines.
//
//	The page is divided into two categories:
//	PHYSICAL PAGE = the physical paper dimensions.
//	LOGICAL PAGE =	Paqe with an origin of (0,0).
//					Size and offset to accomodate the margins.
//
//	All functions use the LOGICAL PAGE.
//	The (x, y) coordinates are offset by the PHYSICAL PAGE when the 
//	WIN32 API is called to print.
//	This allows the page to be laid out with an origin of (0, 0),
//	and separately sized and offset to accomodate the margins.
//
//	The LOGICAL PAGE is further divided into three REGIONS:
//	HEADER
//	BODY
//	FOOTER
//
//	Each region has a PAGEREGIONTYPE struct that describes its size,
//	origin on the LOGICAL PAGE, current point ("cursor"), font, etc.
//
//	Line numbers for each region are 1 indexed.
//
//--------------------------------------------------------------------
//	USING FONTS
//
//	This object can use a different font for each region by passing the
//	logical fonts to the constructor. The lfHeight member function should
//	specify the point size in logical points (note: 72 points per inch).
//		CTextPrinter(	LPCSTR szDocName,	
//						LPLOGFONT BodyFont = NULL,
//						LPLOGFONT HeaderFont = NULL,
//						LPLOGFONT FooterFont = NULL,
//						int iLinesPerHeader = LINESPERHEADER,
//						int iLinesPerFooter = LINESPERFOOTER);
//
//	If a font is NULL or not specified, it uses the default printer font.
//	The default number of lines per header and footer is defined
//	in pagelayout.h.
//
//	NOTE: IF USING THE SAME FONT, the LOGFONT structs should still be separate!
//	For whatever reason, using the same LOGFONT struct for multiple CFont
//	objects can create a problem inside MFC.
//
//	NOTE: If using ChooseFont to select the font, the iPointSize member
//	returns point size in tenths. For example, a 10 point font is returned
//	as 100. Divide by ten before loading the lfHeight member of the LOGFONT.
//	I presume that this is to allow precision to tenths and still
//	use an integer variable.
//
//--------------------------------------------------------------------
//	NAMING CONVENTION
//
//	Where a Protected function exists that also has a Public member
//	function, the Protected function name has a leading and trainling
//	double underscore. For example, the Protected version of NewLine()
//	is __NewLine__(). The Protected functions provide lower level control.
//	This follows a naming convention set by Microsoft, in which lower-level
//	control versions of a function use leading underscores, rather than
//	contriving more spurious names, such as ProtectedNewLine().
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "TextPrinter.h"
#include "resource.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTextPrinter::CTextPrinter(	LPCSTR szDocName,
							LPLOGFONT BodyFont,		// If Fonts are NULL, then
							LPLOGFONT HeaderFont,	// it uses the default
							LPLOGFONT FooterFont, 	// for that printer.
							int iLinesPerHeader,
							int iLinesPerFooter) :
	m_hCDC(NULL),
	m_bOK(FALSE),
	m_bDocStarted(FALSE),
	m_bPageStarted(FALSE),
	m_iVertUnitsLeft(0),
	m_pOldFont(NULL),
	m_iLinesPerHeader(iLinesPerHeader),
	m_iLinesPerFooter(iLinesPerFooter),
	m_iNumPages(0)
{
	HEADER.plfLogFont = NULL;
	BODY.plfLogFont = NULL;
	FOOTER.plfLogFont = NULL;

	//---------------------------------------
	// Ask the user to select a printer.
	// If the user cancels, then skip eveything else.
	CPrintDialog dlg(FALSE);
	if (dlg.DoModal() != IDOK)
	{
		return;			// USER ABORT EXIT <---------------------------
	}


	//---------------------------------------
	// Save the font data for later use
	HEADER.plfLogFont = HeaderFont;
	BODY.plfLogFont = BodyFont;
	FOOTER.plfLogFont = FooterFont;

	if (HeaderFont != NULL)
	{
		HEADER.lfNormal = *HeaderFont;
		HEADER.lfBold = *HeaderFont;
		HEADER.lfItalic = *HeaderFont;
		HEADER.lfBold.lfWeight = FW_BOLD;
		HEADER.lfItalic.lfItalic = TRUE;
	}

	if (FooterFont != NULL)
	{
		FOOTER.lfNormal = *FooterFont;
		FOOTER.lfBold = *FooterFont;
		FOOTER.lfItalic = *FooterFont;
		FOOTER.lfBold.lfWeight = FW_BOLD;
		FOOTER.lfItalic.lfItalic = TRUE;
	}

	if (BodyFont != NULL)
	{
		BODY.lfNormal = *BodyFont;
		BODY.lfBold = *BodyFont;
		BODY.lfItalic = *BodyFont;
		BODY.lfBold.lfWeight = FW_BOLD;
		BODY.lfItalic.lfItalic = TRUE;
	}

	//---------------------------------------
	// User has selected a printer. 
	m_szDeviceName = dlg.GetDeviceName();

	// Create the Device Context.
	m_hCDC = new CDC;
	if (m_hCDC == NULL) 
	{
		return; // FATAL ERROR EXIT <-------------------
	}

	//---------------------------------------
	// Attach the printer device context to the CDC object.
	m_bOK = m_hCDC->Attach(dlg.GetPrinterDC());
	if (!m_bOK)
	{
		return; // FATAL ERROR EXIT <-------------------
	}

	m_hCDC->SetMapMode(MM_TEXT);


	//---------------------------------------
	// Initialize the print job structure and set default header.
	::ZeroMemory(&m_di, sizeof(DOCINFO));
	m_di.cbSize = sizeof(DOCINFO);
	m_di.lpszDocName = szDocName;
	SetHeader(szDocName);
	
	//---------------------------------------
	// Create the fonts and partition the page into
	// the LOGICAL PAGE and REGIONS.
	GetPageDimensions();

	// Initialize the Legend and tab stop array.
	// There is one column at the start of the line, plus one
	// for each tab stop.
	for (int iCol = 0; iCol < NUMCOLS; iCol++)
	{
		m_szLegend[iCol].Empty();
		m_iTabs[iCol] = 0;
	}

	// Get the local time, to print in the header.
	COleDateTime cToday = COleDateTime::GetCurrentTime();

	m_szPrintTime.LoadString(IDS_PRINTDATE);

	// Uses Control Panel settings under Regional and Language Options
	// to set the date format. Uses the Short date format in the local language.
	m_szPrintTime += cToday.Format();

}

//////////////////////////////////////////////////////////////////////
CTextPrinter::~CTextPrinter()
{
	// Cleanup any pending printing activity.
	// NOTE - This is generally how the last page gets printed.
	if (IsSelected())
	{
		// If a font had been passed in, restore the old font.
		RestoreFont();

		// Flow control variables show what activity was still
		// going on.
		if (m_bPageStarted)
		{
			m_hCDC->EndPage();
			m_bPageStarted = FALSE;
		}
		
		if (m_bDocStarted)
		{
			m_hCDC->EndDoc();
			m_bDocStarted = FALSE;
		}

	}

	// De-select any fonts used, then
	// Delete the device context.
	if (m_hCDC != NULL)
	{

		// Now delete the font resources (if any).
		if (HEADER.plfLogFont != NULL)
		{
			HEADER.NormalFont.DeleteObject();
			HEADER.BoldFont.DeleteObject();
			HEADER.ItalicFont.DeleteObject();
		}

		if (FOOTER.plfLogFont != NULL)
		{
			FOOTER.NormalFont.DeleteObject();
			FOOTER.BoldFont.DeleteObject();
			FOOTER.ItalicFont.DeleteObject();
		}

		if (BODY.plfLogFont != NULL)
		{
			BODY.NormalFont.DeleteObject();
			BODY.BoldFont.DeleteObject();
			BODY.ItalicFont.DeleteObject();
		}

		// Finally, delete the device context
		delete m_hCDC;
	}

}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
void CTextPrinter::GetPageDimensions()
{
	//------------------------------------------------------------
	// Get the PHYSICAL text and page dimensions, then divide the
	// page into LOGICAL REGIONS.
	// NOTE - Since we are using the MM_TEXT mapping mode, the 
	// logical units are in pixels.

	if (m_hCDC == NULL) return;		// FATAL ERROR EXIT <-----------

	// Get the total page dimensions and resolution in logical units.
	m_PageSize.cx = m_hCDC->GetDeviceCaps(HORZRES);	// Ask the printer driver.
	m_PageSize.cy = m_hCDC->GetDeviceCaps(VERTRES);

	m_Resolution.cx = m_hCDC->GetDeviceCaps(LOGPIXELSX);	// Units per inch
	m_Resolution.cy = m_hCDC->GetDeviceCaps(LOGPIXELSY);

	// Set Margin sizes in device logical units.
	// Bottom margin is the same as the top margin, and
	// right margin is the same as the left margin.
	m_Margins.cx = (int) ((double) m_Resolution.cx * LEFTMARGIN);
	m_Margins.cy = (int) ((double) m_Resolution.cy * TOPMARGIN);

	// Get the total available vertical space in logical units.
	m_iVertUnitsLeft = m_PageSize.cy - 2 * m_Margins.cy;


	//------------------------------------------------------------
	// Now set sizes and margins for each region.
	// Set the header and footer for two lines each.
	// The body will use all remaining space.
	// Note that each region may have its own font.
	SetRegionDimensions(HEADER, m_iLinesPerHeader);
	SetRegionDimensions(FOOTER, m_iLinesPerFooter);
	SetRegionDimensions(BODY);

	// Offset origins and current point of footer and body,
	// so that they appear in sequence after each other.
	// Body starts right after the header.
	// Footer starts after the body.
	OffsetRegion(BODY, 0, HEADER.PageSize.cy + HEADER.Origin.y);
	OffsetRegion(FOOTER, 0, BODY.PageSize.cy + BODY.Origin.y);

}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	If iNumLines = 0, then it uses all available space on the page.
//	The member variable m_iVertUnitsLeft is used to deduct the height
//	of the lines from the total amount of page space left.
void CTextPrinter::SetRegionDimensions(PAGEREGIONTYPE& Region, int iNumLines)
{
	int	iNumUnitsAsked;

	// Make the font for this region (if any)
	if (Region.plfLogFont != NULL)
	{
		// Scale logical font to the device resolution.
		// Logical size for fonts is always specified in Points.
		// However, the output device needs it in logical units.
		// Resolution is in Pixels per Inch.
		// There are 72 Points per Inch.
		// Therefore...
		double fResolution = -((double) m_Resolution.cy ) / 72.0;
		Region.lfNormal.lfHeight = (long) ((double) Region.lfNormal.lfHeight * fResolution); 
//		Region.lfNormal.lfHeight *= -(m_Resolution.cy / POINTSPERINCH);
		Region.lfItalic.lfHeight = Region.lfNormal.lfHeight;
		Region.lfBold.lfHeight = Region.lfNormal.lfHeight;

		// Create the normal, bold, and italic fonts.
		Region.NormalFont.CreateFontIndirect(&Region.lfNormal);
		Region.BoldFont.CreateFontIndirect(&Region.lfBold);
		Region.ItalicFont.CreateFontIndirect(&Region.lfItalic);
	}

	// Get the line height for this font.
	UseRegionFont(Region);
	m_hCDC->GetOutputTextMetrics(&Region.TextMetric);
	RestoreFont();

	// Adjust total height to be used by the extra line spacing.
	Region.iLineHeight = (int) ((double) Region.TextMetric.tmHeight * (1.0 + LINESPACING));
	
	// Calculate the amount of vertical space needed for this many lines.
	// If 0 lines was passed in, use all remaining available space.
	iNumUnitsAsked = (iNumLines > 0) ? iNumLines * Region.iLineHeight : m_iVertUnitsLeft;

	// Make sure it doesn't go off the page.
	iNumUnitsAsked = (iNumUnitsAsked > m_iVertUnitsLeft) ? m_iVertUnitsLeft : iNumUnitsAsked;

	// Calculate the number of logical units remaining.
	m_iVertUnitsLeft -= iNumUnitsAsked;

	// If the number of lines was not specified, then
	// calculate the number of lines in this region.
	Region.iNumLines = (iNumLines > 0) ? iNumLines : iNumUnitsAsked / Region.iLineHeight;

	// Set the region's Logical dimensions
	Region.PageSize.cx = m_PageSize.cx - 2 * m_Margins.cx;
	Region.PageSize.cy = iNumUnitsAsked;
	
	// Initialize the origin, relative to the LOGICAL PAGE margins.
	// It will be offset later, depending on whether it's the footer or body.
	Region.Origin.x = 0;
	Region.Origin.y = 0;

	// Start on line 1, at the origin.
	Region.iCurrentLine = 1;
	Region.CurrentPoint = Region.Origin;
	
}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	Offset a region by a specified (x, y) amount.
//	Use this to position a region within the LOGICAL PAGE.
void CTextPrinter::OffsetRegion(PAGEREGIONTYPE& Region, int x, int y)
{
	// Offset the origin
	Region.Origin.x += x;
	Region.Origin.y += y;

	// Offset the current point
	Region.CurrentPoint.x += x;
	Region.CurrentPoint.y += y;
}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	This function takes x and y to be LOGICAL PAGE coordinates.
//	This prints text centered on the current line.
void CTextPrinter::__PrintCentered__(LPCSTR szText, PAGEREGIONTYPE& Region)
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------

	// Calculate the width of the text in this font..
	UseRegionFont(Region);
	CString szTemp = szText;
	CSize TextSize = m_hCDC->GetTextExtent(szTemp);
	Region.CurrentPoint.x = (Region.PageSize.cx - TextSize.cx) /2;
	RestoreFont();
	
	// Print the text
	__Print__(szText, Region);

}

//********************************************************************
//		FUNCTIONS
//********************************************************************

//////////////////////////////////////////////////////////////////////
//	Print text from the current point and goes to the next line (CR/LF).
//	This can result in creating a new page, if the line goes off page.
void CTextPrinter::PrintLn(LPCSTR szText)
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------
	if (szText == NULL)  return;

	// Print the text, then give a new line.
	__Print__(szText, BODY);
	NewLine();
}

//////////////////////////////////////////////////////////////////////
//	Prints text from the current point, but does not follow with a newline.
//	Current Point is left at the end of the line.
//	This function ONLY prints in the BODY region.
void CTextPrinter::Print(LPCSTR szText)
{
	if (!m_bOK) return;				// FATAL ERROR EXIT <-----------
	if (szText == NULL)  return;

	// Print the text, but no new line.
	__Print__(szText, BODY);
}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	This function prints text at the Region's Current Point, then
//	updates the current point.
//	The only offsetting done here is the origin for the margins.
void CTextPrinter::__Print__(LPCSTR szText, PAGEREGIONTYPE& Region)
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------

	// Start document, if not already done so.
	// Two Flow Control variables are used to transparently trigger the
	// StartDoc() and StartPage()/EndPage() WIN32 API calls.
	if (!m_bDocStarted)
	{
		// Start the printing
		int iResult = m_hCDC->StartDoc(&m_di);

		// Did a fatal printer error occur?
		if (iResult < 0)
		{
			// MS Documentation specifies negative numbers as the
			// error return for StartDoc.
			// LOG ERROR WITH ERROR MODULE
			m_bOK = FALSE;
			return;			// FATAL ERROR EXIT <--------------------
		}
		else
		{	// Doc started ok.
			m_bDocStarted = TRUE;
		}
		m_iPageNum = 0;
	}

	// Start a new page, if not already done so
	if (!m_bPageStarted)
	{
		NewPage();
	}

	// Finally - offset for the page margins and Print the text!
	int x = Region.CurrentPoint.x + m_Margins.cx;
	int y = Region.CurrentPoint.y + m_Margins.cy;
	m_hCDC->MoveTo(x, y);

	// Start with the default Region font
	FontType	fFont = NORMAL;

	// Print one character at a time.
	// This allows font switching commands to be embedded within the line.
	m_hCDC->SetTextAlign(TA_UPDATECP);
	while (*szText)
	{
		// Check for a font switch command
		switch (*szText)
		{
		case SWITCHTOBOLD:		// Switch to Bold
			fFont = BOLD;
			szText++;
			break;

		case SWITCHTOITALIC:	// Switch to Italic
			fFont = ITALIC;
			szText++;
			break;

		case SWITCHTONORMAL:	// Switch to Normal
			fFont = NORMAL;
			szText++;
			break;

		default:
			break;
		}

		UseRegionFont(Region, fFont);
		m_hCDC->TextOut(x, y, szText, 1);

		// Calculate the width of the text and update the Region's current pointer.
		CString szTemp = *szText;
		CSize TextSize = m_hCDC->GetTextExtent(szTemp);
		Region.CurrentPoint.x += TextSize.cx;

		RestoreFont();
		szText++;

	}
	m_hCDC->SetTextAlign(TA_NOUPDATECP);

}

//////////////////////////////////////////////////////////////////////
//	Prints the header, centered on the page.
//	If szHeader is NULL, it uses the default from SetHeader().
void CTextPrinter::PrintHeader(LPCSTR szHeader)
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------

	// Use default text if none supplied
	if (szHeader == NULL) szHeader = m_szHeader;

	// Print the header, left justfied.
	__Print__(szHeader, HEADER);

	// Print the time/date, right justified.
	UseRegionFont(HEADER);
	CSize iLen = m_hCDC->GetTextExtent(m_szPrintTime);
	HEADER.CurrentPoint.x = HEADER.PageSize.cx - iLen.cx;
	__Print__(m_szPrintTime, HEADER);
	RestoreFont();

	// Print the column legends
	__NewLine__(HEADER);
	PrintLegend();
	__NewLine__(HEADER);

	ScribeLineAbove(0, 9999, &HEADER);
}

//////////////////////////////////////////////////////////////////////
//	Prints the footer, centered on the page.
//	If szFooter is NULL, it uses the default from SetFooter().
void CTextPrinter::PrintFooter(LPCSTR szFooter)
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------

	// Use default text if none supplied
	if (szFooter == NULL) szFooter = m_szFooter;

	__PrintCentered__(szFooter, FOOTER);
	__NewLine__(FOOTER);
}

//////////////////////////////////////////////////////////////////////
//	Sets the default header
void CTextPrinter::SetHeader(LPCSTR szHeader)
{
	if (szHeader != NULL)
	{
		m_szHeader = szHeader;
	}
}

//////////////////////////////////////////////////////////////////////
//	Sets the default footer
void CTextPrinter::SetFooter(LPCSTR szFooter)
{
	if (szFooter != NULL)
	{
		m_szFooter = szFooter;
	}
}

//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	Goes to the start of the next line. Has the effect of a (CR/LF).
//	If the next line goes off the page, then it goes to the top
//	of the next page.
//
//	RETURNS:	TRUE if new line ran off the page and a new page is needed.
//				FALSE if new line is still on page (no new page needed).
BOOL CTextPrinter::__NewLine__(PAGEREGIONTYPE &Region)
{
	BOOL	bRetVal;

	// Carriage return
	Region.CurrentPoint.x = Region.Origin.x;

	// Line feed
	Region.CurrentPoint.y += Region.iLineHeight;
	Region.iCurrentLine++;

	// Check for end of page
	bRetVal = (Region.iCurrentLine > Region.iNumLines);

	return bRetVal;

}

//////////////////////////////////////////////////////////////////////
//	Goes to the start of hte next line, creating a new page if needed.
void CTextPrinter::NewLine()
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------

	BOOL bNeedNewPage = __NewLine__(BODY);

	// Did it run past the end of the page?
	if (bNeedNewPage)
	{
		NewPage();
	}
}

//////////////////////////////////////////////////////////////////////
//	Makes a new page and sets the current pointer to the top.
//	If a previous page existed, it closes the printing for that page
//	before making the new page.
//	It also prints the default Header and Footer.
void CTextPrinter::NewPage()
{
	if (!m_bOK) return;	// FATAL ERROR EXIT <-----------
	int iResult;

	//------------------------------------------------------------------
	// The Flow Control variable is used to see if
	// an existing page is currently printing. If so, it closes the page
	// to complete the print operation.
	// NOTE - This will only be false for the very first print operation.
	if (m_bPageStarted)
	{
		// Close out the old page
		m_hCDC->EndPage();
		m_bPageStarted = FALSE;
	}

	// Start a new page. 
	// Also, start a new doc if that hasn't happened yet.
	if (!m_bDocStarted)
	{
		iResult = m_hCDC->StartDoc(&m_di);
		m_bDocStarted = TRUE;

		// Did a fatal error occur?
		if (iResult < 0)
		{
			// LOG ERROR WITH ERROR MODULE
			m_bOK = FALSE;
			return;				// FATAL ERROR EXIT <-----------------------
		}
	}

	iResult = m_hCDC->StartPage();

	//------------------------------------------------------------------
	// Did a fatal printer error occur?
	if (iResult < 0)
	{
		// LOG ERROR WITH ERROR MODULE
		m_bOK = FALSE;
		return;				// FATAL ERROR EXIT <-----------------------
	}

	//------------------------------------------------------------------
	// Setup the new page and reset cursors to the region origins.
	m_bPageStarted = TRUE;
	HEADER.CurrentPoint = HEADER.Origin;
	HEADER.iCurrentLine = 1;

	FOOTER.CurrentPoint = FOOTER.Origin;
	FOOTER.iCurrentLine = 1;

	BODY.CurrentPoint = BODY.Origin;
	BODY.iCurrentLine = 1;
	
	m_iPageNum++;

	// Format the footer string.
	CString szFooter;
	if (m_iNumPages > 0)
	{
		szFooter.Format(IDS_FOOTERPGNOFN, m_iPageNum, m_iNumPages);
	}
	else
	{
		szFooter.Format(IDS_FOOTERPGN, m_iPageNum);
	}

	SetFooter(szFooter);

	// Ensure that the default header and footer are printed immediately..
	// Otherwise, the last page could end before getting to the footer.
	PrintHeader(NULL);
	PrintFooter(NULL);

}

//////////////////////////////////////////////////////////////////////
//	Prints centered text in the BODY on the current line.
void CTextPrinter::PrintCentered(LPCSTR szText)
{
	if (!m_bOK) return;				// FATAL ERROR EXIT <-----------
	if (szText == NULL)  return;

	// Print centered on the current line
	__PrintCentered__(szText, BODY);

}

//////////////////////////////////////////////////////////////////////
//	Tabs current point in the BODY to the new Horiz position, 
//	in TABSIZE of inches from the left margin.
//	pagelayout.h defines TABSIZE as 10, so this function
//	tabs in tenths of inches.
void CTextPrinter::TabToHorz(int x)
{
	// Inches per pixel times number of inches.
	BODY.CurrentPoint.x = (m_Resolution.cx * x)/TABSIZE;
}

//////////////////////////////////////////////////////////////////////
//	Tabs to the given line number, or to the bottom of the page if it's off the page.
void CTextPrinter::TabToLine(int y)
{
	// Don't let it go off the page.
	y = (y > BODY.iNumLines) ? BODY.iNumLines : y;

	// Number of total units for the lines.
	BODY.CurrentPoint.y = (y - 1) * BODY.iLineHeight;
	BODY.iCurrentLine = y;
}

//////////////////////////////////////////////////////////////////////
//	Draws a horizontal line on the current line from iFromX to iToX,
//	but not beyond the right margin.
//	iFromX and iToX are in inches relative to the left margin.
//	If iToX is left blank, it assumes the right margin.
//	If iFromX is left blank, is draws from the left to the right margin.
void CTextPrinter::ScribeLineAbove(int iFromX, int iToX, PAGEREGIONTYPE* pRegion)
{
	if (!m_bOK) return;				// FATAL ERROR EXIT <-----------

	// Calculate the (X, Y) locations
	CPoint	cFrom;
	CPoint	cTo;

	// If no region specified, use the BODY region
	if (pRegion == NULL) pRegion = &BODY;

	// Ascend 1/10 of a line upwards for an overscore.
	// ASCENT is defined in pagelayout.h
	int iAscent = pRegion->iLineHeight / ASCENT;

	// Offset for the vertical margin of the PHYSICAL PAGE
	cFrom.y = pRegion->CurrentPoint.y - iAscent + m_Margins.cy;
	cTo.y = cFrom.y;

	// Calculate the From and To points on the PHYSICAL PAGE, 
	// offsetting for the horz margin.
	cFrom.x = iFromX * m_Resolution.cx + m_Margins.cx;
	cTo.x = iToX * m_Resolution.cx + m_Margins.cx;

	// Clip line to the margins
	int iLimit = m_PageSize.cx - m_Margins.cx;
	cTo.x = (cTo.x > iLimit) ? iLimit : cTo.x;
	cFrom.x = (cFrom.x > iLimit) ? iLimit : cFrom.x;

	// Draw the line.
	m_hCDC->MoveTo(cFrom);
	m_hCDC->LineTo(cTo);
}


//////////////////////////////////////////////////////////////////////
//	PROTECTED FUNCTION
//	Sets the current font to the region's font.
//	If the region does not have a specific font , then
//	this function just returns without taking action.
void CTextPrinter::UseRegionFont(PAGEREGIONTYPE &Region, FontType fType)
{
	// Do error checking first.
	if (m_hCDC == NULL) return;				// FATAL ERROR EXIT -------
	if (Region.plfLogFont == NULL) return;	// EXIT - REGION does not have own font
	if (m_pOldFont != NULL) return;			// FATAL ERROR, Font already in use!

	CFont* pFont;

	// Select the font, storing the old font.
	switch (fType)
	{
	case NORMAL:
		pFont = &Region.NormalFont;
		break;

	case ITALIC:
		pFont = &Region.ItalicFont;
		break;

	case BOLD:
		pFont = &Region.BoldFont;
		break;

	default:
		pFont = &Region.NormalFont;
		break;
	}

	m_pOldFont = m_hCDC->SelectObject(pFont);
}

//////////////////////////////////////////////////////////////////////
//	Restores the printer after using a Region's font.
//	PROTECTED FUNCTION
void CTextPrinter::RestoreFont()
{
	if (m_hCDC == NULL) return;		// FATAL ERROR EXIT ----------------
	if (m_pOldFont == NULL) return;	// EXIT - No old font to restore

	// Restore the old font.
	m_hCDC->SelectObject(m_pOldFont);
	m_pOldFont = NULL;
}

//////////////////////////////////////////////////////////////////////
//	Prints a tabbed legend as part of the header.
//	PROTECTED FUNCTION
void CTextPrinter::PrintLegend()
{
	UseRegionFont(HEADER);

	// Legend appears on the second line of the header.
	int y = HEADER.Origin.y + m_Margins.cy + HEADER.iLineHeight;

	// There can be a legend for each column.
	for (int iCol = 0; iCol < NUMCOLS; iCol++)
	{
		int x = HEADER.Origin.x + m_Margins.cx + m_iTabs[iCol];
		m_hCDC->TextOut(x, y, m_szLegend[iCol]);
	}

	RestoreFont();
}

//////////////////////////////////////////////////////////////////////
void CTextPrinter::SetLegend(LPCSTR szLegend, int iCol)
{
	// Set the legend for this column number
	if (szLegend != NULL)
	{
		// Make sure we didn't set the column number too high.
		// There is one column for the first row, plus one for each tab.
		// The first column is column 0, the last column is NUMTABS.
		iCol = (iCol < NUMCOLS) ? iCol : NUMTABS;
		
		m_szLegend[iCol] = szLegend;
	}
}

//////////////////////////////////////////////////////////////////////
void CTextPrinter::SetTabStop(int iTab, int iCol)
{
		// Make sure we didn't set the column number too high.
		// There is one column for the first row, plus one for each tab.
		// The first column is column 0, the last column is NUMTABS.
		iCol = (iCol < NUMCOLS) ? iCol : NUMTABS;

		// Set the tab stop, in logical units, given the
		// distance in tenths of inches.
		m_iTabs[iCol] = iTab * m_Resolution.cx / TABSIZE;
}
