//////////////////////////////////////////////////////////////////////
// Questionlist.cpp: implementation of the CQuestionList class.
//
//	PURPOSE
//		This implements the business rules for the SERIM model.
//	
//		A CQuestionList object contains an array of CQuestion objects.
//
//	ROLE
//		Information Holder and Structurer.
//		This class creates a list of blank SERIM questions and
//		allows iteration through the list. The client object uses
//		this to get a pointer to a particular CQuestion object.
//
//		Since only this class is allowed to either instansiate or
//		delete CQuestion objects, the CQuestion objects can only be
//		created via a class factory, known only to this class.
//		The only information that the client ever controls is the
//		response to a question.
//
//	ASSUMPTIONS:
//		The SERIM Model is fixed and not subject to change.
//		There are always 81 questions.
//		The text of the questions never changes.
//		The interpretation of the responses never changes.
//
//	ORDER OF CALCULATIONS:
//		As desired, set the priorities via:
//			SetCostPriority(), SetTechPriority, and SetSchedPriority().
//
//		After all questions have been answered...
//		1) Calculate PA4 through PA13		- Calculate(EquationType eType)
//		2) Calculate PB through PL			- Calculate(EquationType eType)
//		3) Calculate PA1, PA2, PA3, PN, PO	- CalculateSummary(EquationType eType)
//		4) Calculate the total risk	PA		- CalculateTotal()
//
//		The CalculateAll() function does all of those calculations in sequence,
//		storing the results for retrieval via GetResult(EquationType eType).
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "questionlist.h"
#include "resource.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
//	CQuestion class factory, declared here so that only CQuestionList
//	can instantiate or delete CQuestion objects.
//	The class factory itself is never instantiated.
//	The static functions are used instead.
//////////////////////////////////////////////////////////////////////
class CQuestionFactory : public CQuestion
{
public:
	// Cannot call "new" here, because the CQuestion Constuctor is protected.
	// The implementation of "new" cannot access the protected Constructor.
	// Therefore, we call a "Create" function, which in turn does the "new".
	// The same resoning applies to the object destruction.
	static CQuestion* Create(QuestionType* qType) { return CQuestion::Create(qType); };
	static Destroy(CQuestion* q) { CQuestion::Destroy(q); };

private:
	// The factory is never instansiated.
	// Only the static members are ever called.
	CQuestionFactory(QuestionType* qType) : CQuestion(qType) {};
	virtual ~CQuestionFactory() {};
};

//////////////////////////////////////////////////////////////////////
//	Static structures to define a question. The interface has no business
//	knowing this, so they are defined here and not in the header file.
//////////////////////////////////////////////////////////////////////

// Define the data for each equation type.
struct EquationDataStruct {
		EquationType	eType;
		UINT			uFlags;
		double*			pdWeight;
		double			dAnswer;
		UINT			uTextID;
};

// Arrays of SERIM weightings for various equations
static double	PA1weights[] = { 0.043, 0.043, 0.087, 0.087, 0.087, 0.130, 0.130, 0.130, 0.130, 0.130 };
static double	PA2weights[] = { 0.136, 0.136, 0.136, 0.136, 0.090, 0.090, 0.045, 0.045, 0.045, 0.136 };
static double	PA3weights[] = { 0.136, 0.136, 0.136, 0.136, 0.090, 0.090, 0.045, 0.045, 0.045, 0.136 };
static double	PNweights[]  = { 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.040, 0.040, 0.040, 0.125 };
static double	POweights[]  = { 0.045, 0.045, 0.045, 0.045, 0.140, 0.140, 0.140, 0.140, 0.140, 0.140 };

#define NUMWEIGHTS sizeof(PA1weights) / sizeof(double)

// Defines the equation, weighting (if any), text description, and the current value.
// ASSUMPTION - Data is defined in the same order as the EquationType enum.
// That lets the EquationType be used as a index without having to search the array.
static struct EquationDataStruct EquationData[] = {
  // Name	Category, Phase		Weighting	Answer	String ID
  //		or Activity for		  for
  //		Calculate()		CalculateSummary()
	PA,		0,					NULL,		0,	IDS_PA,		// Total Risk
	PA1,	0,					PA1weights,	0,	IDS_PA1,	// Summary - Technical Risk
	PA2,	0,					PA2weights,	0,	IDS_PA2,	// Summary - Cost Risk
	PA3,	0,					PA3weights,	0,	IDS_PA3,	// Summary - Schedule Risk

	PA4,	C_ORGANIZATION,		NULL,		0,	IDS_PA4,	// Category Risks
	PA5,	C_ESTIMATION,		NULL,		0,	IDS_PA5,
	PA6,	C_MONITORING,		NULL,		0,	IDS_PA6,
	PA7,	C_DEVELOPMENT,		NULL,		0,	IDS_PA7,
	PA8,	C_TOOLS,			NULL,		0,	IDS_PA8,
	PA9,	C_RISKCULTURE,		NULL,		0,	IDS_PA9,
	PA10,	C_USABILITY,		NULL,		0,	IDS_PA10,
	PA11,	C_CORRECTNESS,		NULL,		0,	IDS_PA11,
	PA12,	C_RELIABILITY,		NULL,		0,	IDS_PA12,
	PA13,	C_PERSONNEL,		NULL,		0,	IDS_PA13,

	PB,		P_PREREQUIREMENTS,	NULL,		0,	IDS_PB,		// Phase Risks
	PC,		P_REQUIREMENTS,		NULL,		0,	IDS_PC,
	PD,		P_DESIGN,			NULL,		0,	IDS_PD,
	PE,		P_CODE,				NULL,		0,	IDS_PE,
	PF,		P_TESTING,			NULL,		0,	IDS_PF,
	PG,		P_MAINTENANCE,		NULL,		0,	IDS_PG,

	PH,		A_IDENTIFICATION,	NULL,		0,	IDS_PH,		// Activity Risks
	PI,		A_PLANNING,			NULL,		0,	IDS_PI,
	PJ,		A_ASSESSMENT,		NULL,		0,	IDS_PJ,
	PK,		A_MITIGATION,		NULL,		0,	IDS_PK,
	PL,		A_REPORTING,		NULL,		0,	IDS_PL,
	PM,		A_PREDICTION,		NULL,		0,	IDS_PM,

	PN,		0,					PNweights,	0,	IDS_PN,		// Summary - Process Risk
	PO,		0,					POweights,	0,	IDS_PO,		// Summary - Product Risk
};

//////////////////////////////////////////////////////////////////////
// Define the questions and their category, phases, and activities, and their UI object.
static struct QuestionType QuestionArray[] = {
	// O - Organization
	IDS_O1, C_ORGANIZATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O2, C_ORGANIZATION | P_PREREQUIREMENTS | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O3, C_ORGANIZATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PLANNING | A_ASSESSMENT | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O4, C_ORGANIZATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O5, C_ORGANIZATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O6, C_ORGANIZATION | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O7, C_ORGANIZATION | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_O8, C_ORGANIZATION | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,

	// E - Estimation
	IDS_E1, C_ESTIMATION | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | UI_LISTBOX,
	IDS_E2, C_ESTIMATION | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_E3, C_ESTIMATION | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_E4, C_ESTIMATION | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_E5, C_ESTIMATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PLANNING | A_REPORTING | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_E6, C_ESTIMATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_E7, C_ESTIMATION | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PLANNING | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,

	// M - Monoitoring
	IDS_M1, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PLANNING | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M2, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M3, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M4, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M5, C_MONITORING | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M6, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_M7, C_MONITORING | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_REPORTING | A_PREDICTION | UI_RADIOBUTTONS,

	// DM - Development
	IDS_DM1, C_DEVELOPMENT | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM2, C_DEVELOPMENT | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM3, C_DEVELOPMENT | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM4, C_DEVELOPMENT | P_REQUIREMENTS | P_DESIGN | P_CODE | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM5, C_DEVELOPMENT | P_TESTING | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM6, C_DEVELOPMENT | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_DM7, C_DEVELOPMENT | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,

	// T - Tools
	IDS_T1, C_TOOLS | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T2, C_TOOLS | P_DESIGN | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T3, C_TOOLS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T4, C_TOOLS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T5, C_TOOLS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T6, C_TOOLS | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T7, C_TOOLS | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T8, C_TOOLS | P_CODE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_T9, C_TOOLS | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,

	// RC - Risk Culture
	IDS_RC1,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC2,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC3,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC4,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC5,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC6,  C_RISKCULTURE | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC7,  C_RISKCULTURE | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC8,  C_RISKCULTURE | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC9,  C_RISKCULTURE | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC10, C_RISKCULTURE | P_PREREQUIREMENTS | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_RC11, C_RISKCULTURE | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PLANNING | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,

	// U - Usability
	IDS_U1, C_USABILITY | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_U2, C_USABILITY | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_U3, C_USABILITY | P_REQUIREMENTS | A_IDENTIFICATION | A_ASSESSMENT | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_U4, C_USABILITY | P_DESIGN | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_U5, C_USABILITY | P_REQUIREMENTS | A_IDENTIFICATION | A_MITIGATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_U6, C_USABILITY | P_DESIGN | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,

	// C - Correctness
	IDS_C1, C_CORRECTNESS | P_REQUIREMENTS | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C2, C_CORRECTNESS | P_DESIGN | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C3, C_CORRECTNESS | P_CODE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C4, C_CORRECTNESS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C5, C_CORRECTNESS | P_PREREQUIREMENTS | P_REQUIREMENTS | P_DESIGN | P_CODE | P_TESTING | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_C6, C_CORRECTNESS | P_CODE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C7, C_CORRECTNESS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_C8, C_CORRECTNESS | P_MAINTENANCE | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_C9, C_CORRECTNESS | P_TESTING | A_IDENTIFICATION | A_ASSESSMENT | A_PREDICTION | UI_RADIOBUTTONS,
	
	// R - Reliability
	IDS_R1,  C_RELIABILITY | P_DESIGN | P_CODE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R2,  C_RELIABILITY | P_DESIGN | P_CODE | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,
	IDS_R3,  C_RELIABILITY | P_REQUIREMENTS | P_DESIGN | P_CODE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R4,  C_RELIABILITY | P_CODE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R5,  C_RELIABILITY | P_DESIGN | P_CODE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R6,  C_RELIABILITY | P_CODE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R7,  C_RELIABILITY | P_TESTING | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | UI_RADIOBUTTONS,
	IDS_R8,  C_RELIABILITY | P_MAINTENANCE | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_R9,  C_RELIABILITY | P_TESTING | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_R10, C_RELIABILITY | P_TESTING | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_R11, C_RELIABILITY | P_TESTING | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_R12, C_RELIABILITY | P_TESTING | A_IDENTIFICATION | A_PREDICTION | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	
	// P - Personnel
	IDS_P1, C_PERSONNEL | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | A_PLANNING | A_ASSESSMENT | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_P2, C_PERSONNEL | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_P3, C_PERSONNEL | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_P4, C_PERSONNEL | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | A_MITIGATION | UI_RADIOBUTTONS,
	IDS_P5, C_PERSONNEL | P_PREREQUIREMENTS | A_IDENTIFICATION | A_PREDICTION | UI_RADIOBUTTONS,

};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CQuestionList::CQuestionList() : 
	m_dSchedPriority(0.333),
	m_dCostPriority(0.333),
	m_dTechPriority(0.333),
	m_iNumQuestions(0),
	m_bmodified(FALSE)
{
	// Confirm that all the questions were entered.
	// SERIM model is fixed at 81 questions.
	ASSERT(NUMQUESTIONS == (sizeof(QuestionArray) / sizeof(QuestionType) ) );

	// Confirm that all the equations were entered.
	// SERIM model is fixed at 28 eqations.
	ASSERT(NUMEQUATIONS == (sizeof(EquationData) / sizeof(EquationDataStruct) ) );

	// Create the list of questions.
	// FOR each question in the array DO
	for (int i = 0; i < NUMQUESTIONS; i++)
	{
		// IF a question is defined THEN
		if (QuestionArray[m_iNumQuestions].uQuestionID > 0)
		{
			// Create the CQuestion object and add it to the list.
			m_QuestionList[i] = CQuestionFactory::Create(&QuestionArray[m_iNumQuestions]);

			// Next question
			m_iNumQuestions++;
		}
		else
		// ELSE
		{
			// NULL the CQuestion*
			m_QuestionList[i] = NULL;
		}
		// END IF
	}
	// END FOR

	// Clear the answer array
	for (i = 0; i < NUMEQUATIONS; i++) EquationData[i].dAnswer = 0;

}

//////////////////////////////////////////////////////////////////////
CQuestionList::~CQuestionList()
{
	// Destroy the questions.
	// FOR each question object DO
	for (int i = 0; i < NUMQUESTIONS; i++)
	{
		// IF a question object exists THEN
		if (m_QuestionList[i] != NULL)
		{
			// Delete the question object
			CQuestionFactory::Destroy( m_QuestionList[i] );
			m_QuestionList[i] = NULL;
		}
		// END IF
	}
	// END FOR
}

//////////////////////////////////////////////////////////////////////
//	RETURNS:	CQuestion* for question #iNum, or
//				NULL if iNum is ann invalid question number.
CQuestion* CQuestionList::GetQuestion(UINT iNum)
{ 
	if (iNum < m_iNumQuestions)
		return m_QuestionList[iNum]; 
	else
		return NULL;
}

//////////////////////////////////////////////////////////////////////
//	Calculates the complete SERIM set of equations and stores
//	the results in the answer array.
//	The answers can be obtained by the GetResult() method.
void CQuestionList::CalculateAll()
{
	// First, calculate the categories, activities, and phases,
	for(int i = 0; i < NUMEQUATIONS; i++)
	{
		Calculate(EquationData[i].eType);
	}

	// Next, calculate the summaries of the preceeding equations.
	for(i = 0; i < NUMEQUATIONS; i++)
	{
		CalculateSummary(EquationData[i].eType);
	}

	// Finally, calculate the total.
	CalculateTotal();
}

//********************************************************************
//********************************************************************

//////////////////////////////////////////////////////////////////////
//	RETURNS:	Value of the answers for the given category, phase, or activity.
//				This implements the SERIM equations:
//					P(A4) through P(A13)
//					P(B) through P(M)
//				Zero if the equation type is invalid.
//
//	ASSUMPTION:	Assumes that all questions in the particular equation set
//				have been answered.
//
double CQuestionList::Calculate(EquationType eqType)
{
	double	dResult = 0;
	int		iTotal = 0;

	//-------------------------------------------------------------------
	// Get the flag for this equation type.
	UINT uFlags = EquationData[eqType].uFlags;

	//-------------------------------------------------------------------
	// Now calculate the average for each question containing this flag.
	// FOR each question DO
	for (UINT i = 0; i < m_iNumQuestions; i++)
	{
		// IF the question is in this category THEN
		if (m_QuestionList[i]->HasFlag( uFlags ) )
		{
			// Add the answer to the running total
			dResult += m_QuestionList[i]->GetAnswerValue();
			iTotal++;
		}
		// END IF
	}
	// END FOR

	//-------------------------------------------------------------------
	// Store and Return the average
	dResult /= (double) iTotal;
	EquationData[eqType].dAnswer = dResult; 
	return dResult;
}

//////////////////////////////////////////////////////////////////////
//	RETURNS:	Weighted summary of the given equation, or
//				Zero if the equation is not a summary equation.
//
//	ASSUMPTION:	Assumes that the equations PA4 through PA13 have already been
//				calculated via Calculate().
//
double CQuestionList::CalculateSummary(EquationType eqType)
{
	double	dResult = 0;
	double*	pWeights = EquationData[eqType].pdWeight;

	// If this equation type is a summary of others, calc the summary
	if (pWeights != NULL)
	{
		// FOR each equation in the summary DO
		for (int i = 0; i < NUMWEIGHTS; i++)
		{
			// Multiple the value by the weighting
			dResult += pWeights[i] * EquationData[PA4 + i].dAnswer;
		}
		// END FOR

		// Save the result in the array.
		EquationData[eqType].dAnswer = dResult;
	}

	return dResult;
}

//////////////////////////////////////////////////////////////////////
//	RETURNS:	Weighted summary of the SERIM equations.
//
//	ASSUMPTION:	Assumes that CalculateSummary() has been called for
//				each equation.
//
double CQuestionList::CalculateTotal()
{
	double dResult = EquationData[PA1].dAnswer * m_dTechPriority;
	dResult += EquationData[PA2].dAnswer * m_dCostPriority;
	dResult += EquationData[PA3].dAnswer * m_dSchedPriority;

	EquationData[PA].dAnswer = dResult;
	return dResult;
}

//////////////////////////////////////////////////////////////////////
//	RETURNS:	Text string for the equation's description.
//
LPCSTR CQuestionList::GetEquationText(EquationType eType)
{
	static	CString strText;
	strText.LoadString(EquationData[eType].uTextID);
	return strText;
}

//////////////////////////////////////////////////////////////////////
//	RETURNS:	Current value of the given equation
//
//	ASSUMPTION:	Assumes that the equation has been calculated.
//
double	CQuestionList::GetResult(EquationType eqType) 
{ 
	double dAnswer = EquationData[eqType].dAnswer; 
	return dAnswer;
};


//////////////////////////////////////////////////////////////////////
// RETURNS:	The Equation type for the specified category.
//
EquationType CQuestionList::GetEquationType(UINT uCategory)
{
	EquationType	eType = PA;

	// SCAN the EquationData array for this category
	for (UINT i = 0; i < NUMEQUATIONS; i++)
	{
		if (EquationData[i].uFlags & uCategory)
		{
			eType = EquationData[i].eType;
			break;
		}
	}

	return eType;
}

//////////////////////////////////////////////////////////////////////
void CQuestionList::ClearAll()
{
	// FOR each question
	for (UINT i = 0; i < NUMQUESTIONS; i++)
	{
		// Clear the response
		m_QuestionList[i]->SetAnswerOrdinal(0);
	}
	// END FOR

	// Reset the equation results
	for (i = 0; i < NUMEQUATIONS; i++)
	{
		EquationData[i].dAnswer = 0;
	}

	// Reset the weightings
	m_dCostPriority = 0.333;
	m_dSchedPriority = 0.333;
	m_dTechPriority = 0.333;

	// Clear the modified flags
	ClearModified();
}

//////////////////////////////////////////////////////////////////////
BOOL CQuestionList::IsModified()
{
	// The 81 quesdtions share a common static modified flag, so if
	// any question was modified, then all will show up as modified.
	// The only other modified flag to check is our own flag
	// for the Weightings.

	if (m_bmodified == TRUE) return TRUE;

	if (m_QuestionList[0]->IsModified()) return TRUE;

	return FALSE;

}

//////////////////////////////////////////////////////////////////////
void CQuestionList::ClearModified()
{
	// The 81 quesdtions share a common static modified flag, so 
	// clearing one questionwill clear them all.
	// The only other modified flag is our own flag
	// for the Weightings.
	m_bmodified = FALSE;
	m_QuestionList[0]->ClearModified();
}
